;******************************************************
;******************************************************
;
; Main Project     : OCRG
; Project Name     : OCRG CWVM - CW Voltmeter
; File Name        : cwvm_main.asm
; Version          : A03
; PIC Used         : PIC12HV615
; Assembler        : MPLAB MPASMWIN from Microchip Technology, Inc.
; Initial Date     : Nov 19, 2009
; Last Update date : Dec 22, 2009
; Firmware Eng.    : WA7ZVY
; HW Design Eng.   : WA7ZVY
; Copyright 2009   : WA7ZVY
;
;******************************************************
;
; Description of this code:
;
; This code starts execution upon the release of the
; power on reset condition. The code continually
; executes the main routine and executes the
; interrupt code upon an interrupting condition.
;
;	1) reset_vector
;	2) interrupt_vector, interrupt dispatcher
;	3) main loop
;
; ******************************************************
;
; Function List (in top-2-bottom order):
;   void interrupt(void);
;   void main(void);
;
;******************************************************
;
;   Modification History:
;   =-=-=-=-=-=-=-=-=-=-=
;   11-19-09 pm - Initial creation.
;
;******************************************************

;*****************************************************
;   === Include files ===
;*****************************************************
	LIST   P=12HV615			;List directive to define processor

#include	P12HV615.INC		;Processor specific variable definitions
#include	constants.inc		;Coding constants
#include	micro_bits.inc		;Hardware defined bits
#include	variable_def.inc	;Variable definitions


;*****************************************************
;       === Configuration Word Definitions ===
;*****************************************************
; Defines for PIC12HV615 operation in the CWVM Board

; '__CONFIG' directive is used to embed configuration data within .asm file.
; The labels following the directive are located in the respective .inc file.
; See respective data sheet for additional information on configuration word.
	__config _INTOSCIO & _IOSCFS_8MHZ & _CP_OFF & _BOR_OFF & _PWRTE_ON & _WDT_OFF & _MCLRE_ON

; '__idlocs' sets the four ID locations to the hexadecimal value of expression.
; These are not readable by the cpu, but can be read by the programning hardware.
;	__idlocs	0x1234


;**********************************************************************

RESET_VECTOR	ORG	0x0000	; processor reset vector location
;	nop						; for icd debug
	pagesel	cwvm_main		;
    goto    cwvm_main		; go to beginning of program



;*****************************************************
;*****************************************************
; 	Name:	interrupt
;	Abstract:
;	This is the interrupt handler. It is entered
;	whenever there is an interrupt asserted.
;	It looks for the source of the interrupt
;	and dispatches to the appropriate interrupt
;	handling routine.
;	This routine is the top level interrupt code (ie. it
;	calls other routines which return to it, but
;	it is never called by other routines).
;
;	Sources of interrupts are:
;	1). PIC USART
;		a). Timer 0
;
;	Callable Routines:  None.
;
;	Called Routines:
		extern	timer0_irq_service
;		extern	timer1_irq_service
;		extern	timer2_irq_service
;
;	Global Variables Used:
;		w_temp
;		status_temp
;		pclath_temp
;
;	Local Variables Used:  None.
;		
;	Return Values:	None.
;		
;*****************************************************
;*****************************************************
INTERRUPT_VECTOR	ORG	0x0004	; processor interrupt vector location
; save current state
    movwf   w_temp          ; save off current W register contents
    movf    STATUS,W        ; move status register into W register
    movwf   status_temp     ; save off contents of STATUS register
    movf    PCLATH,W        ; move pclath register into w register
    movwf   pclath_temp     ; save off contents of PCLATH register

; call isr subroutines if an interrupt is pending
	banksel	BANK0
	pagesel	isr_1
	btfss	INTCON, T0IF
	goto	isr_1
	pagesel	timer0_irq_service
	call	timer0_irq_service

isr_1
;	banksel	BANK0
;	pagesel	isr_2
;	btfss	PIR1, TMR1IF
;	goto	isr_2
;	pagesel	timer1_irq_service
;	call	timer1_irq_service

isr_2
;	banksel	BANK0
;	pagesel	isr_2
;	btfss	PIR1, TMR2IF
;	goto	isr_3
;	pagesel	timer1_irq_service
;	call	timer1_irq_service

isr_3
; restore previous state	
    movf    pclath_temp,W   ; retrieve copy of PCLATH register
    movwf   PCLATH          ; restore pre-isr PCLATH register contents
    movf    status_temp,W   ; retrieve copy of STATUS register
    movwf   STATUS          ; restore pre-isr STATUS register contents
    swapf   w_temp,F
    swapf   w_temp,W        ; restore pre-isr W register contents
    retfie                  ; return from interrupt



;*****************************************************
;   === Define the Morse Code Character Table ===
;*****************************************************
; Format is: Number of elements, "-" = Dash, "." = Dot, " " = Character Space
; NOTE!  All of this resides in the first 256 words of code space, therefore
; there is no need to worry about crossing a 256 word boundary.
	global	morse_table
	global	morse_one,  morse_two,    morse_three,    morse_four, morse_five
	global	morse_six,  morse_seven,  morse_eight,    morse_nine, morse_zero
	global	morse_v,    morse_period, morse_question, morse_error

morse_table
morse_one
	addwf	PCL,F
	dt	0x07, ".----  "
morse_two
	addwf	PCL,F
	dt	0x07, "..---  "
morse_three
	addwf	PCL,F
	dt	0x07, "...--  "
morse_four
	addwf	PCL,F
	dt	0x07, "....-  "
morse_five
	addwf	PCL,F
	dt	0x07, ".....  "
morse_six
	addwf	PCL,F
	dt	0x07, "-....  "
morse_seven
	addwf	PCL,F
	dt	0x07, "--...  "
morse_eight
	addwf	PCL,F
	dt	0x07, "---..  "
morse_nine
	addwf	PCL,F
	dt	0x07, "----.  "
morse_zero
	addwf	PCL,F
	dt	0x07, "-----  "

morse_v
	addwf	PCL,F
	dt	0x06, "...-  "

morse_period
	addwf	PCL,F
	dt	0x08, ".-.-.-  "

morse_question
	addwf	PCL,F
	dt	0x08, "..--..  "

morse_error
	addwf	PCL,F
	dt	0x0a, "........  "


;*****************************************************
;   === Define the Morse Code Speed Table ===
;*****************************************************
; Format is: Speed Value (the value gets put into the PR2 register)
; NOTE!  All of this resides in the first 256 words of code space, therefore
; there is no need to worry about crossing a 256 word boundary.
	global	speed_table

speed_table
	addwf	PCL,F
	dt	0x1c		;43.4 WPM
	dt	0x1d		;40.4 WPM
	dt	0x1f		;37.8 WPM
	dt	0x21		;35.5 WPM
	dt	0x24		;32.6 WPM
	dt	0x27		;30.0 WPM
	dt	0x2b		;27.9 WPM
	dt	0x2f		;25.5 WPM
	dt	0x34		;22.5 WPM
	dt	0x3b		;20.2 WPM
	dt	0x43		;17.8 WPM
	dt	0x4e		;15.0 WPM
	dt	0x5e		;12.6 WPM
	dt	0x75		;10.0 WPM
	dt	0x9c		; 7.5 WPM
	dt	0xea		; 5.0 WPM


CWVM_MAIN_CODE	CODE			;cwvm_main code space

;*****************************************************
;*****************************************************
;
;	Name:		cwvm_main
;	
;	Abstract:
;		This is the main entry point for
;		code execution on the CWVM board.
;		This routine is the highest level code (ie. it
;		calls other routines which return to it, but
;		it is never called by other routines).
;
;	Callable Routines:  None.
;
;	Called Routines:
		extern	init_hardware, delay_100msec, binary_to_decimal, send_morse_char
		extern	divide_by_n, multiply_by_n, adc_sample_delay
;
;	Global Variables Used:
;		
;	Local Variables Used:
;	
;	Return Values:	N/A.
; 		
;*****************************************************
;*****************************************************
cwvm_main
	pagesel	init_hardware		;Initialize the various hardware functions
	call	init_hardware
	banksel	BANK0

	movlw	FALSE				;Initialize various software control values
	movwf	b0_ToneOn			;Set the CW tone to off
	movwf	b0_OverRange		;Set the Over Range flag to FALSE

	movlw	0x05				;Do a 500 millisecond delay to let "the button" stop bouncing
	movwf	arg1_lo
	pagesel	delay_100msec
	call	delay_100msec
	banksel	BANK0

cm_1
; Read the Speed potentiometer's wiper pin voltage level to determine the Morse code speed setting
	banksel	BANK0
	bcf		INTCON,GIE			;GLOBAL DISABLE ALL INTERRUPTS (prevents sleep exit from vectoring us)
	bcf		PIR1,ADIF			;Clear the ADC interrupt flag
	movlw	CFG_ADCON0_CH0		;Right justify result, Vdd for Vref, use AN0, turn on ADC
	movwf	ADCON0
	banksel	BANK1
	bsf		PIE1,ADIE			;Enable the Analog to Digital Converter Interrupt
	bsf		INTCON,PEIE			;Enable the Peripheral Interrupts
	pagesel	adc_sample_delay	;Delay to allow ADC sample cap to stabilize
	call	adc_sample_delay
	banksel	BANK0
	bsf		ADCON0,GO			;Start an ADC conversion
	sleep						;Go to sleep to make a nice quiet system during the conversion
;	Execution continues here after the ADC completes its conversion
	movf	ADRESH,W			;Get the high two bits of the conversion
	movwf	arg1_hi
	banksel	BANK1
	movf	ADRESL,W			;Get the low byte of the conversion
	movwf	arg1_lo
	bcf		INTCON,PEIE			;Disable the Peripheral Interrupts
	banksel	BANK1
	bcf		PIE1,ADIE			;Disable the Analog to Digital Converter Interrupt
	banksel	BANK0
	bcf		ADCON0,GO			;Force the ADC to be done, this shouldn't be needed, but just do it!
	bcf		PIR1,ADIF			;Clear the ADC interrupt flag

;	pageselw	speed_table		;Set the page for looking up the speed value in the speed_table
	movlw	0x00				;Set the page for looking up the speed value in the speed_table
	movwf	PCLATH				;Set the page for looking up the speed value in the speed_table

	bcf		STATUS,C			;Shift the ADC's upper most 4 bits so we can reference the speed_table
	rrf		arg1_hi,F
	rrf		arg1_lo,F
	bcf		STATUS,C
	rrf		arg1_hi,F
	rrf		arg1_lo,F
	nop
	swapf	arg1_lo,W			;The bottom 4 bits now contain the index into the speed_table
	andlw	0x0f				;And mask off any high order bits
	call	speed_table			;On return, W contains the value to put into PR2
	banksel	BANK1
	movwf	PR2
	banksel	BANK0
	movlw	CFG_T2CON			;Set Timer 2's prescaler and postscaler both to 16
	movwf	T2CON

; Read the 0.6V reference voltage level to calibrate the Voltmeter Correction Factor
	banksel	BANK0
	bcf		INTCON,GIE			;GLOBAL DISABLE ALL INTERRUPTS (prevents sleep exit from vectoring us)
	bcf		PIR1,ADIF			;Clear the ADC interrupt flag
	movlw	CFG_ADCON0_V06		;Right justify result, Vdd for Vref, read Vref 0.6V, turn on ADC
	movwf	ADCON0
	banksel	BANK1
	bsf		PIE1,ADIE			;Enable the Analog to Digital Converter Interrupt
	bsf		INTCON,PEIE			;Enable the Peripheral Interrupts
	pagesel	adc_sample_delay	;Delay to allow ADC sample cap to stabilize
	call	adc_sample_delay
	banksel	BANK0
	bsf		ADCON0, GO			;Start an ADC conversion
	sleep						;Go to sleep to make a nice quiet system during the conversion
;	Execution continues here after the ADC completes its conversion
	movf	ADRESH,W			;Get the high two bits of the conversion
	movwf	b0_correction_hi	;And put it into the correction factor
	banksel	BANK1
	movf	ADRESL,W			;Get the low byte of the conversion
	banksel	BANK0
	movwf	b0_correction_lo
	bcf		INTCON,PEIE			;Disable the Peripheral Interrupts
	banksel	BANK1
	bcf		PIE1,ADIE			;Disable the Analog to Digital Converter Interrupt
	banksel	BANK0
	bcf		ADCON0,GO			;Force the ADC to be done, this shouldn't be needed, but just do it!
	bcf		PIR1,ADIF			;Clear the ADC interrupt flag

; CODE TO REPORT THE VREF(0.6V) A/D READING
#IF(REPORT_VREF == TRUE)	
	movf	b0_correction_hi,W
	movwf	arg1_hi
	movf	b0_correction_lo,W
	movwf	arg1_lo
	goto	bypass_code_1
#ENDIF
#IF(REPORT_VREF == FALSE)	;This keeps the amount of code the same either way
	nop
	nop
	nop
	nop
	nop
#ENDIF
; END OF CODE TO REPORT THE VREF(0.6V) A/D READING

;	Compare the reference voltage read with the ideal "typical" result value
;		(((0.575V/5.000V)*1024)-1) = 117. = 0x0075
	bsf		b0_correct_negative, 0x00	;Assume the correction needed will be a negative subtraction
	movf	b0_correction_hi,W
	movwf	b0_tmp1_hi
	movf	b0_correction_lo,W
	movwf	b0_tmp1_lo
	movlw	0x75				;The low byte of the ideal result value
	subwf	b0_tmp1_lo,F
	btfsc	STATUS,C			;Check for a borrow condition on the low byte
	goto	cm_3a
	decf	b0_tmp1_hi,F		;A borrow occurred, so account for the borrow
cm_3a
	movlw	0x00				;The high byte of the ideal result value
	subwf	b0_tmp1_hi,F
	nop
	btfss	b0_tmp1_hi,0x07		;Check to see if the high byte went negative...
	goto	cm_3b
	clrf	b0_correct_negative	;Yes, the correction will actually need to be a positive addition
	comf	b0_tmp1_lo,F		;Complement the values to use for the addition
	comf	b0_tmp1_hi,F

;	Multiply the correction value by 8.7 (5.000V/0.575V) to obtain the full scale correction value
cm_3b
	movf	b0_tmp1_lo,W		;First multiply it by 7
	movwf	arg1_lo
	movf	b0_tmp1_hi,W
	movwf	arg1_hi
	movlw	0x07
	movwf	arg2_lo
	clrf	arg2_hi
	pagesel	multiply_by_n
	call	multiply_by_n		;Result returned in arg3_wrd
	movf	arg3_lo,W			;Now divide that by 10 to get the 0.7 part
	movwf	arg1_lo
	movf	arg3_hi,W
	movwf	arg1_hi
	movlw	0x0a
	movwf	arg2_lo
	clrf	arg2_hi
	pagesel	divide_by_n
	call	divide_by_n			;Result returned in arg3_wrd
	banksel	BANK0
	bcf		STATUS,C			;Now do the multiply by 8 part
	rlf		b0_tmp1_lo,F
	rlf		b0_tmp1_hi,F
	bcf		STATUS,C
	rlf		b0_tmp1_lo,F
	rlf		b0_tmp1_hi,F
	bcf		STATUS,C
	rlf		b0_tmp1_lo,F
	rlf		b0_tmp1_hi,F
	movf	arg3_lo,W			;Now add the multiplied by 0.7 part to the multiplied by 8 part
	addwf	b0_tmp1_lo,F
	btfsc	STATUS,C			;Check for a carry
	incf	b0_tmp1_hi,F		;Account for the carry
	movf	arg3_hi,W
	addwf	b0_tmp1_hi,F
; CORRECTION FACTOR CODE ENABLEMENT
#IF(ZERO_CORRECTION == FALSE)
	movf	b0_tmp1_lo,W
	movwf	b0_correction_lo
	movf	b0_tmp1_hi,W
	movwf	b0_correction_hi	;Now, we are all set for doing the correction later on
#ENDIF
#IF(ZERO_CORRECTION == TRUE)
	movlw	0x00
	movwf	b0_correction_lo
	movlw	0x00
	movwf	b0_correction_hi	;Now any correction factor is effectively eliminated
#ENDIF
; END OF CORRECTION FACTOR CODE ENABLEMENT

; Read the Voltmeter pin voltage level
	banksel	BANK0
	bcf		INTCON,GIE			;GLOBAL DISABLE ALL INTERRUPTS (prevents sleep exit from vectoring us)
	bcf		PIR1,ADIF			;Clear the ADC interrupt flag
	movlw	CFG_ADCON0_CH3		;Right justify result, Vdd for Vref, use AN3, turn on ADC
	movwf	ADCON0
	banksel	BANK1
	bsf		PIE1,ADIE			;Enable the Analog to Digital Converter Interrupt
	bsf		INTCON,PEIE			;Enable the Peripheral Interrupts
	pagesel	adc_sample_delay	;Delay to allow ADC sample cap to stabilize
	call	adc_sample_delay
	banksel	BANK0
	bsf		ADCON0, GO			;Start an ADC conversion
	sleep						;Go to sleep to make a nice quiet system during the conversion
;	Execution continues here after the ADC completes its conversion
	movf	ADRESH,W			;Get the high two bits of the conversion
	movwf	arg1_hi
	banksel	BANK1
	movf	ADRESL,W			;Get the low byte of the conversion
	movwf	arg1_lo
	bcf		INTCON,PEIE			;Disable the Peripheral Interrupts
	banksel	BANK1
	bcf		PIE1,ADIE			;Disable the Analog to Digital Converter Interrupt
	banksel	BANK0
	bcf		ADCON0,GO			;Force the ADC to be done, this shouldn't be needed, but just do it!
	bcf		PIR1,ADIF			;Clear the ADC interrupt flag

								;Setup for doing the correction factor adjustments
								;Correction to add or subtract =
								;   ((ADC Value Read * Full Scale Correction Value) / 1024.)
	movf	b0_correction_lo,W
	movwf	arg2_lo
	movf	b0_correction_hi,W
	movwf	arg2_hi
	pagesel	multiply_by_n
	call	multiply_by_n		;Result returned in arg3_wrd
	banksel	BANK0
	movf	arg3_lo,W
	movwf	arg1_lo
	movf	arg3_hi,W
	movwf	arg1_hi
	clrf	arg2_lo				;Now setup to divide the previously multiplied result by 1024
	movlw	0x04
	movwf	arg2_hi
	pagesel	divide_by_n
	call	divide_by_n			;We now have the scaled correction value to use in arg3_wrd
	banksel	BANK0
	movf	ADRESH,W			;Get the high two bits of the conversion, again
	movwf	arg1_hi
	banksel	BANK1
	movf	ADRESL,W			;Get the low byte of the conversion, again
	movwf	arg1_lo
	banksel	BANK0

	btfss	b0_correct_negative, 0x00
	goto	cm_5
						;if(CORRECT_SIGN == NEGATIVE)
	movf	arg3_lo,W
	subwf	arg1_lo,F
	btfss	STATUS,C
	decf	arg1_hi,F			;Account for the borrow if necessary
	movf	arg3_hi,W
	subwf	arg1_hi,F
	goto	cm_6
						;endif

cm_5
						;if(CORRECT_SIGN == POSITIVE)
	movf	arg3_lo,W
	addwf	arg1_lo,F
	btfsc	STATUS,C
	incf	arg1_hi,F			;Account for the carry if necessary
	movf	arg3_hi,W
	addwf	arg1_hi,F
						;endif

cm_6
	movlw	0xfc				;Check for an overrange condition
	andwf	arg1_hi,W
	btfss	STATUS,Z
	bsf		b0_OverRange, 0x00	;Flag an overrange condition (refer to send_it routine)

;		Multiply the value by 2.44 to scale it to "volts.hundredths_of_volts"
;		Register usage:
;			b0_tmp1_wrd = x2
;			b0_tmp2_wrd = x4
;			b0_tmp3_wrd = x0.04
;			b0_tmp4_wrd = x0.4

	bcf		STATUS,C			;Multiply by 2
	rlf		arg1_lo,F
	rlf		arg1_hi,F
	movf	arg1_hi,W
	movwf	b0_tmp1_hi
	movf	arg1_lo,W
	movwf	b0_tmp1_lo

	bcf		STATUS,C			;Once more to multiply it by 4
	rlf		arg1_lo,F
	rlf		arg1_hi,F
	movf	arg1_hi,W
	movwf	b0_tmp2_hi
	movf	arg1_lo,W
	movwf	b0_tmp2_lo

	movlw	0x64				;Divide it by 100 to get x0.04 part
	movwf	arg2_lo
	clrf	arg2_hi
	pagesel	divide_by_n
	call	divide_by_n			;Result returned in arg3_wrd
	movf	arg3_hi,W			;Save the x0.04 result
	movwf	b0_tmp3_hi
	movf	arg3_lo,W
	movwf	b0_tmp3_lo

	movf	b0_tmp2_hi,W		;Do the x0.4 part
	movwf	arg1_hi
	movf	b0_tmp2_lo,W
	movwf	arg1_lo
	movlw	0x0a				;Divide it by 10 to get x0.4 part
	movwf	arg2_lo
	clrf	arg2_hi
	pagesel	divide_by_n
	call	divide_by_n			;Result returned in arg3_wrd
	movf	arg3_hi,W			;Save the x0.4 result
	movwf	b0_tmp4_hi
	movf	arg3_lo,W
	movwf	b0_tmp4_lo

;		Now add up all of the pieces to get the entire voltage value
	clrf	arg1_hi
	clrf	arg1_lo

	movf	b0_tmp1_lo,W
	addwf	arg1_lo,F
	btfsc	STATUS,C
	incf	arg1_hi,F
	movf	b0_tmp1_hi,W
	addwf	arg1_hi,F

	movf	b0_tmp3_lo,W
	addwf	arg1_lo,F
	btfsc	STATUS,C
	incf	arg1_hi,F
	movf	b0_tmp3_hi,W
	addwf	arg1_hi,F

	movf	b0_tmp4_lo,W
	addwf	arg1_lo,F
	btfsc	STATUS,C
	incf	arg1_hi,F
	movf	b0_tmp4_hi,W
	addwf	arg1_hi,F
;		arg1_wrd now has the binary voltage reading

bypass_code_1	;ENTRY POINT LABEL TO LEARN WHAT THE CORRECTION FACTOR VALUE IS (SEE WAY ABOVE)

	pagesel	binary_to_decimal	;Convert the binary value to decimal nibbles
	call	binary_to_decimal	;The 4 nibble result returned in arg2_wrd
	banksel	BANK0
	movf	arg2_hi,W			;Save the 4 nibbles for now
	movwf	b0_nibbles_hi
	movf	arg2_lo,W
	movwf	b0_nibbles_lo


; Send the morse code characters to report the ADC value to the user
	banksel	BANK0
	bsf		INTCON, T0IE		;Enable Timer 0 Interrupt
	nop
	bsf		INTCON, PEIE		;Enable the Peripheral Interrupts
	nop
	bsf		INTCON, GIE			;GLOBAL ENABLE OF ALL OF THE ABOVE INTERRUPTS!

	swapf	b0_nibbles_hi,W		;Get the most significant nibble (4th nibble)
	andlw	0x0f
	btfsc	STATUS,Z			;Check if its zero (we don't send the 4th nibble if its a zero)
	goto	cm_7

	movwf	arg1_lo
	pagesel	send_morse_char		;Send the morse code character for the decimal digit value
	call	send_morse_char
	banksel	BANK0

cm_7
	movf	b0_nibbles_hi,W		;Get the 3rd nibble
	andlw	0x0f
	movwf	arg1_lo
	pagesel	send_morse_char		;Send the morse code character for the decimal digit value
	call	send_morse_char
	banksel	BANK0

	movlw	0x0f				;Send a decimal point (we use 0x0f to indicate a ".")
	movwf	arg1_lo
	pagesel	send_morse_char		;Send the morse code character for a "."
	call	send_morse_char
	banksel	BANK0

	swapf	b0_nibbles_lo,W		;Get the 2nd nibble
	andlw	0x0f
	movwf	arg1_lo
	pagesel	send_morse_char		;Send the morse code character for the decimal digit value
	call	send_morse_char
	banksel	BANK0

	movf	b0_nibbles_lo,W		;Get the least significant nibble (1st nibble)
	andlw	0x0f
	movwf	arg1_lo
	pagesel	send_morse_char		;Send the morse code character for the decimal digit value
	call	send_morse_char
	banksel	BANK0

	movlw	0x0e				;Send a "V" for volts (we use 0x0e to indicate a "V")
	movwf	arg1_lo
	pagesel	send_morse_char		;Send the morse code character for the decimal digit value
	call	send_morse_char
	banksel	BANK0

	clrf	b0_OverRange		;Clear the over-range flag

	clrf	INTCON				;Disable ALL interrupts

; Wait around for 2 seconds and then do it all again
	movlw	0x14
	movwf	arg1_lo
	pagesel	delay_100msec
	call	delay_100msec
	banksel	BANK0
	pagesel	cm_1
	goto	cm_1


	end

; End of File (main.asm)
