;******************************************************
;******************************************************
;
; Main Project     : VFOpm
; Project Name     : VFOpm - Digital Variable Frequency Oscillator
; File Name        : vfo_main.asm
; Version          : A01
; PIC Used         : PIC16F648A
; Assembler        : MPLAB MPASMWIN from Microchip Technology, Inc.
; Initial Date     : Nov. 4, 2014
; Last Update date : Nov. 4, 2014
; Firmware Eng.    : WA7ZVY
; HW Design Eng.   : WA7ZVY
;
;******************************************************
;
; Description of this code:
;
; This code starts execution upon the release of the
; power on reset condition. The code continually
; executes the vfo_main routine.
;
;******************************************************
;
;   Modification History:
;   =-=-=-=-=-=-=-=-=-=-=
;   20141104 pm - Initial creation.
;
;******************************************************

;*****************************************************
;   === Include files ===
;*****************************************************
	LIST   P=16F648A			;List directive to define processor

#include	P16F648A.INC		;Processor specific variable definitions
#include	constants.inc		;Coding constants
#include	micro_bits.inc		;Hardware defined bits
#include	variable_def.inc	;Variable definitions


;*****************************************************
;       === Configuration Word Definitions ===
;*****************************************************
; Defines for PIC16F648A operation in the Digital VFO Board

; '__CONFIG' directive is used to embed configuration data within .asm file.
; The labels following the directive are located in the respective .inc file.
; See respective data sheet for additional information on configuration word.

	__CONFIG	_BODEN_OFF & _BOREN_OFF & _CP_OFF & _DATA_CP_OFF & _PWRTE_OFF & _WDT_OFF & _LVP_OFF & _MCLRE_ON & _INTOSC_OSC_NOCLKOUT

; '__idlocs' sets the four ID locations to the hexadecimal value of expression.
; These are not readable by the cpu, but can be read by the programning hardware.

	__idlocs	0x1234

;**********************************************************************

RESET_VECTOR	ORG	0x0000	; processor reset vector location
;	nop						; for icd debug
    goto    vfo_main		; go to beginning of program

;	This code contains goto statements that are located
;	at the beginning of each page of processor memory
;	(except page0).  These goto statements send
;	run-amuck code execution back through the reset
;	vector location in page0 at location 0.  All unused
;	program memory locations are filled with goto 0x0000
;	instructions.
CRASH_VECTOR_PAGE1	ORG	0x0800
	pagesel	RESET_VECTOR
	goto	RESET_VECTOR


;*****************************************************
INTERRUPT_VECTOR	ORG	0x0004	; processor interrupt vector location
    goto	RESET_VECTOR		; no interrupts from this hardware
;*****************************************************

std_dsply
	addwf	PCL,F
	dt		"FREQ:xx.xxx.xxxSTEP:xx.xxx.xxx"

base_message
	addwf	PCL,F
	dt		"NEW BASE STORED"


;----------------------------------
FREQ_TABLE_LOW	ORG	0x0100		;Start the frequency table on the 2nd 256 byte boundary
;----------------------------------
FREQ_TABLE_O	;Frequency Table for Ones of Hertz, LSB is lowest addr, MSB is highest addr
	addwf	PCL,F
	dt		0x00			;0Hz
	dt		0x00
	dt		0x00
	dt		0x00

	dt		0x6b			;1Hz
	dt		0x00
	dt		0x00
	dt		0x00

	dt		0xd7			;2Hz
	dt		0x00
	dt		0x00
	dt		0x00

	dt		0x42			;3Hz
	dt		0x01
	dt		0x00
	dt		0x00

	dt		0xad			;4Hz
	dt		0x01
	dt		0x00
	dt		0x00

	dt		0x19			;5Hz
	dt		0x02
	dt		0x00
	dt		0x00

	dt		0x84			;6Hz
	dt		0x02
	dt		0x00
	dt		0x00

	dt		0xf0			;7Hz
	dt		0x02
	dt		0x00
	dt		0x00

	dt		0x5b			;8Hz
	dt		0x03
	dt		0x00
	dt		0x00

	dt		0xc6			;9Hz
	dt		0x03
	dt		0x00
	dt		0x00

;----------------------------------
FREQ_TABLE_T				;Frequency Table for Tens of Hertz
	addwf	PCL,F
	dt		0x00			;00Hz
	dt		0x00
	dt		0x00
	dt		0x00

	dt		0x32			;10Hz
	dt		0x04
	dt		0x00
	dt		0x00

	dt		0x63			;20Hz
	dt		0x08
	dt		0x00
	dt		0x00

	dt		0x95			;30Hz
	dt		0x0c
	dt		0x00
	dt		0x00

	dt		0xc7			;40Hz
	dt		0x10
	dt		0x00
	dt		0x00

	dt		0xf9			;50Hz
	dt		0x14
	dt		0x00
	dt		0x00

	dt		0x2a			;60Hz
	dt		0x19
	dt		0x00
	dt		0x00

	dt		0xdc			;70Hz
	dt		0x15
	dt		0x00
	dt		0x00

	dt		0x8e			;80Hz
	dt		0x21
	dt		0x00
	dt		0x00

	dt		0xc0			;90Hz
	dt		0x25
	dt		0x00
	dt		0x00

;----------------------------------
FREQ_TABLE_H				;Frequency Table for Hundreds of Hertz
	addwf	PCL,F
	dt		0x00			;000Hz
	dt		0x00
	dt		0x00
	dt		0x00

	dt		0xf1			;100Hz
	dt		0x29
	dt		0x00
	dt		0x00

	dt		0xe3			;200Hz
	dt		0x53
	dt		0x00
	dt		0x00

	dt		0xd4			;300Hz
	dt		0x7d
	dt		0x00
	dt		0x00

	dt		0xc6			;400Hz
	dt		0xa7
	dt		0x00
	dt		0x00

	dt		0xb7			;500Hz
	dt		0xd1
	dt		0x00
	dt		0x00

	dt		0xa9			;600Hz
	dt		0xfb
	dt		0x00
	dt		0x00

	dt		0x9a			;700Hz
	dt		0x25
	dt		0x01
	dt		0x00

	dt		0x8b			;800Hz
	dt		0x4f
	dt		0x01
	dt		0x00

	dt		0x7d			;900Hz
	dt		0x79
	dt		0x01
	dt		0x00

;----------------------------------
FREQ_TABLE_K				;Frequency Table for Kilo-hertz
	addwf	PCL,F
	dt		0x00			;0KHz
	dt		0x00
	dt		0x00
	dt		0x00

	dt		0x6e			;1KHz
	dt		0xa3
	dt		0x01
	dt		0x00

	dt		0xdc			;2KHz
	dt		0x46
	dt		0x03
	dt		0x00

	dt		0x4b			;3KHz
	dt		0xea
	dt		0x04
	dt		0x00

	dt		0xb9			;4KHz
	dt		0x8d
	dt		0x06
	dt		0x00

	dt		0x27			;5KHz
	dt		0x31
	dt		0x08
	dt		0x00

	dt		0x95			;6KHz
	dt		0xd4
	dt		0x09
	dt		0x00

	dt		0x03			;7KHz
	dt		0x78
	dt		0x0b
	dt		0x00

	dt		0x71			;8KHz
	dt		0x1b
	dt		0x0d
	dt		0x00

	dt		0xe0			;9KHz
	dt		0xbe
	dt		0x0e
	dt		0x00

;----------------------------------
FREQ_TABLE_TK				;Frequency Table for Tens of Kilo-hertz
	addwf	PCL,F
	dt		0x00			;00KHz
	dt		0x00
	dt		0x00
	dt		0x00

	dt		0x4e			;10KHz
	dt		0x62
	dt		0x10
	dt		0x00

	dt		0x9c			;20KHz
	dt		0xc4
	dt		0x20
	dt		0x00

	dt		0xe9			;30KHz
	dt		0x26
	dt		0x31
	dt		0x00

	dt		0x37			;40KHz
	dt		0x89
	dt		0x41
	dt		0x00

	dt		0x85			;50KHz
	dt		0xeb
	dt		0x51
	dt		0x00

	dt		0xd3			;60KHz
	dt		0x4d
	dt		0x62
	dt		0x00

	dt		0x21			;70KHz
	dt		0xb0
	dt		0x72
	dt		0x00

	dt		0x6f			;80KHz
	dt		0x12
	dt		0x83
	dt		0x00

	dt		0xbc			;90KHz
	dt		0x74
	dt		0x93
	dt		0x00

;----------------------------------
FREQ_TABLE_HK				;Frequency Table for Hundreds of Kilo-hertz
	addwf	PCL,F
	dt		0x00			;000KHz
	dt		0x00
	dt		0x00
	dt		0x00

	dt		0x0a			;100KHz
	dt		0xd7
	dt		0xa3
	dt		0x00

	dt		0x14			;200KHz
	dt		0xae
	dt		0x47
	dt		0x01

	dt		0x1f			;300KHz
	dt		0x85
	dt		0xeb
	dt		0x01

	dt		0x29			;400KHz
	dt		0x5c
	dt		0x8f
	dt		0x02

	dt		0x33			;500KHz
	dt		0x33
	dt		0x33
	dt		0x03

	dt		0x3d			;600KHz
	dt		0x0a
	dt		0xd7
	dt		0x03

	dt		0x48			;700KHz
	dt		0xe1
	dt		0x7a
	dt		0x04

	dt		0x52			;800KHz
	dt		0xb8
	dt		0x1e
	dt		0x05

	dt		0x5c			;900KHz
	dt		0x8f
	dt		0xc2
	dt		0x05


;----------------------------------
FREQ_TABLE_HIGH	ORG	0x0200		;Continue the frequency table on the 3rd 256 byte boundary
;----------------------------------
FREQ_TABLE_M				;Frequency Table for Mega-hertz
	addwf	PCL,F
	dt		0x00			;0MHz
	dt		0x00
	dt		0x00
	dt		0x00

	dt		0x66			;1MHz
	dt		0x66
	dt		0x66
	dt		0x06

	dt		0xcd			;2MHz
	dt		0xcc
	dt		0xcc
	dt		0x0c

	dt		0x33			;3MHz
	dt		0x33
	dt		0x33
	dt		0x13

	dt		0x9a			;4MHz
	dt		0x99
	dt		0x99
	dt		0x19

	dt		0x00			;5MHz
	dt		0x00
	dt		0x00
	dt		0x20

	dt		0x66			;6MHz
	dt		0x66
	dt		0x66
	dt		0x26

	dt		0xcd			;7MHz
	dt		0xcc
	dt		0xcc
	dt		0x2c

	dt		0x33			;8MHz
	dt		0x33
	dt		0x33
	dt		0x33

	dt		0x9a			;9MHz
	dt		0x99
	dt		0x99
	dt		0x39

;----------------------------------
FREQ_TABLE_TM				;Frequency Table for Tens of Mega-hertz
	addwf	PCL,F
	dt		0x00			;00MHz
	dt		0x00
	dt		0x00
	dt		0x00

	dt		0x00			;10MHz
	dt		0x00
	dt		0x00
	dt		0x40

	dt		0x00			;20MHz
	dt		0x00
	dt		0x00
	dt		0x80

	dt		0x00			;30MHz
	dt		0x00
	dt		0x00
	dt		0xc0


	dt		0x00			;[40MHz]
	dt		0x00
	dt		0x00
	dt		0x00

	dt		0x00			;[50MHz]
	dt		0x00
	dt		0x00
	dt		0x00

	dt		0x00			;[60MHz]
	dt		0x00
	dt		0x00
	dt		0x00

	dt		0x00			;[70MHz]
	dt		0x00
	dt		0x00
	dt		0x00

	dt		0x00			;[80MHz]
	dt		0x00
	dt		0x00
	dt		0x00

	dt		0x00			;[90MHz]
	dt		0x00
	dt		0x00
	dt		0x00

;----------------------------------

;*****************************************************
DEEPROM	ORG 0x2100		;Burn into EEPROM the initial default Base Frequency (5.00MHz) for VFO mode
	de	0x00			;Location 0x00 has the LSB of the Base Frequency
	de	0x00
	de	0x00
	de	0x20			;Location 0x03 has the MSB of the Base Frequency

;**********************************************************************



MAIN_CODE	CODE				;Main code space

;*****************************************************
;*****************************************************
;
;	Name:		vfo_main
;	
;	Abstract:
;		This is the main entry point for code execution
;		on the digital vfo oscillator board.
;		This routine is the highest level code (ie. it
;		calls other routines which return to it, but
;		it is never called by other routines).
;
;	Callable Routines:  None.
;
;	Called Routines:
		extern	init_hardware
		extern	delay_50uSec
		extern	delay_2mSec
		extern	delay_N50mSec
		extern	read_eeprom_location, write_eeprom_location
;
;	Global Variables Used:
;		
;	Local Variables Used:
;	
;	Return Values:	N/A.
; 		
;*****************************************************
;*****************************************************
vfo_main
	call	init_hardware		;Initialize the various hardware functions
	banksel	BANK0


;**********************************
;* Initialize all one time options.
;**********************************
	clrf	b0_cstep		;Set current Step to 100KHz.
	bsf		b0_cstep,5

;***************************
;* Load LCD messages to ram:
;***************************
	movlw	b0_mfreq		;Setup indirection register to point at frequency RAM
	movwf	FSR
	bankisel	b0_mfreq
	clrf	b0_tmp1_lo		;clear a pointer
	pageselw	std_dsply
idis
	movf	b0_tmp1_lo,W
	call	std_dsply		;On return W has the character pointed to
	movwf	INDF			;Store the character into the frequency RAM
	incf	FSR,F
	incf	b0_tmp1_lo,F
	nop
	movf	b0_tmp1_lo,W
	xorlw	0x1e			;Done printing the FREQ and STEP messages to RAM?
	btfss	STATUS,Z
	goto	idis			;Jump if more to do

;**********************************
;* Get initial state of optical switch.
;**********************************
	movf	PORTA,W			;Read initial settings of vfo dial
	andlw	0x03			;Isolate VFO bits
	movwf	b0_opto_now		;Initialize the opto switch readings
	movwf	b0_opto_prev	;

;**********************************
;* Get initial state of mode switch.
;**********************************
	movf	PORTA,W			;Read initial setting of mode switch
	andlw	0x08			;Isolate mode switch bit
	movwf	b0_mode_now		;Initialize the mode switch readings
	movwf	b0_mode_prev	;

;**********************
;* Initialize Lcd panel:
;**********************
#IF LCD_TYPE == VARITRONICS
	movlw	0x01			;Setup for doing 50mSec delays
	movwf	arg2_lo

	movlw	0x03			;Set it to 8 bit data bus mode!!! Must do this before
	call	inst			; going to 4 bit mode or it doesn't work!
	call	delay_N50mSec
	movlw	0x03
	call	inst
	call	delay_N50mSec
	movlw	0x03
	call	inst
	call	delay_N50mSec
	movlw	0x02			;Set it to 4 bit data bus with 2 line LCD
	call	inst
	call	delay_N50mSec
	movlw	0x02			;Set it to 4 bit data bus with 2 line LCD
	call	inst
	call	delay_N50mSec
	movlw	0x08
	call	inst
	call	delay_N50mSec
	movlw	0x00			;Clear the display and home the cursor
	call	inst
	call	delay_N50mSec
	movlw	0x01
	call	inst
	call	delay_N50mSec

;*******************************
;* LCD "DISPLAY ON/OFF" FUNCTION COMMAND:
;* -----------------------------
;* LCD REGISTER PORT BITS:
;* D D D D D D D D
;* 7 6 5 4 3 2 1 0
;*
;* n R E R L L L L
;* c S   W 7 6 5 4
;* -----------------------------
;* -----------------------------
;* LCD COMMAND (sent via LCD REGISTER PORT):
;* 0 0 0 0 1 x x x   (x=1=on, x=0=off)
;*         : : : :
;*         : : : Blink
;*         : : Cursor
;*         : Display
;*         Display On/Off Command Bit
;********************************

	movlw	0x00			;First nibble display command
	call	inst			;Send 1st nibble
	call	delay_N50mSec	;Long delays for initial commands
	movlw	0x0c			;Display on, cursor off, blink off
	call	inst			;Send 2nd nibble
	call	delay_N50mSec
#ENDIF

#IF	LCD_TYPE == LUMEX
							;Kick start the display by sending some 8-bit commands!!!
	movlw	0x03			;Function Set Command = 8-bit xfer mode
	call	inst
	call	delay_50uSec
	movlw	0x03			;Function Set Command = 8-bit xfer mode
	call	inst
	call	delay_50uSec
	movlw	0x02			;Function Set Command = 4-bit xfer mode
	call	inst
	call	delay_50uSec

							;Now setup the display for the real 4-bit operating mode that we want
	movlw	0x02			;Function Set Command = 4-bit xfer mode
	call	inst
	movlw	0x0c			;Function Set Command = 2-line mode, 5x11 characters
	call	inst
	call	delay_50uSec

	movlw	0x00			;Display Control Command
	call	inst
	movlw	0x0c			;Display Control Command = Display ON, Cursor OFF, Blink OFF
	call	inst
	call	delay_50uSec

	movlw	0x00			;Display Clear Command
	call	inst
	movlw	0x01
	call	inst
	call	delay_2mSec

	movlw	0x00			;Entry Mode Set Command
	call	inst
	movlw	0x06			;Entry Mode Set Command = Increment Mode, Shift of Entire Display OFF
	call	inst
	call	delay_50uSec
#ENDIF

	call	zero			;Set frequency to "zero" and also send to Harris DDS
	call	dfreq			;Display the current frequency in LCD
	call	dstep			;Display the current frequency step in LCD


;**************************
;* READ OPTO SWITCH and STORE BASE SWITCH:
;**************************
switch
	pagesel	switch			;All of the code resides in this one page
	banksel	BANK0			;All registers used by the code are found in BANK0
	btfss	PORTA,0x04		;Check if STORE BASE switch is pushed in (grounded)
	call	storebase		;Store new base frequency if STORE BASE switch is pushed in

switch_1
 	movf	PORTA,W			;Read mode switch value to determine if we are in DDS or VFO mode
	andlw	0x08			;Isolate the mode switch bit
	movwf	b0_mode_now		;Store current mode switch state
	xorwf	b0_mode_prev,W	;Is it the same as the previous read of the mode?
	btfsc	STATUS,Z		;Branch if isn't
	goto	switch_2		;Its the same, so jump as there is no need to update display

	movlw	0x02			;Do a 100mSec delay and check again... debounce the mode switch
	movwf	arg2_lo
	call	delay_N50mSec
	movf	PORTA,W			;Read the mode switch again for another check
	andlw	0x08
	movwf	b0_mode_now
	xorwf	b0_mode_prev,W
	btfsc	STATUS,Z
	goto	switch_2
	movlw	0x02			;Do another 100mSec delay and check again... debounce the mode switch
	movwf	arg2_lo
	call	delay_N50mSec
	movf	PORTA,W			;And one more final check before committing to a mode change
	andlw	0x08
	movwf	b0_mode_now
	xorwf	b0_mode_prev,W
	btfsc	STATUS,Z
	goto	switch_2
	call	digits_to_ddsbits	;Convert decimal digits to dds binary bits
	call	addbase			;Add base vfo, if needed

	call	shift			;Shift it into the Harris DDS numerically controlled oscillator
	call	dfreq			;Since the mode has changed, update the displayed freq
	call	dstep			; and the displayed step
	movf	b0_mode_now,W	;Put the current mode state value
	movwf	b0_mode_prev	; into the previous mode state value for the next time through

switch_2
	btfss	PORTA,0x02		;Check if VFO button is pushed in to change step (grounded)
	call	bump			;Bump step down (or up) if VFO button is pushed in
	movf	PORTA,W			;Read switch input pins
	andlw	0x03			;Isolate the switch bits
	movwf	b0_opto_now		;Store current switch state
	movf	b0_opto_prev,W	;Load old switch data from last read
	xorlw	0x03			;Was previous read a 11
	btfsc	STATUS,Z
	goto	st3				;Yes, then jump
	movf	b0_opto_prev,W	;Reload old switch data from last read
	xorlw	0x02			;Was previous read a 10
	btfsc	STATUS,Z
	goto	st2				;Yes, then jump
	movf	b0_opto_prev,W	;Reload old switch data from last read
	xorlw	0x01			;Was previous read a 01
	btfsc	STATUS,Z
	goto	st1				;Yes, then jump
							;Otherwise just fall through for a previous read of 00

;****************************
;* Original sw=00
;* By knowing the values
;* of two sucsessive reads
;* from the VFO Switch we can
;* determine what direction,
;* and if, the switch is
;* being twisted.
;****************************
st0
	movf	b0_opto_now,W	;Save the current read to previous for next time
	movwf	b0_opto_prev
	xorlw	0x01			;Is it an up operation?
	btfsc	STATUS,Z
	goto	up
	movf	b0_opto_now,W
	xorlw	0x02			;Is it a down operation?
	btfsc	STATUS,Z
	goto	dn
	goto	switch			;Otherwise go back to switch routine

;****************
;* Original sw=01
;****************
st1
	movf	b0_opto_now,W	;Save the current read to previous for next time
	movwf	b0_opto_prev
	xorlw	0x03			;Is it an up operation?
	btfsc	STATUS,Z
	goto	up
	movf	b0_opto_now,W
	xorlw	0x00			;Is it a down operation?
	btfsc	STATUS,Z
	goto	dn
	goto	switch			;Otherwise go back to switch routine

;****************
;* Original sw=10
;****************
st2
	movf	b0_opto_now,W	;Save the current read to previous for next time
	movwf	b0_opto_prev
	xorlw	0x00			;Is it an up operation?
	btfsc	STATUS,Z
	goto	up
	movf	b0_opto_now,W
	xorlw	0x03			;Is it a down operation?
	btfsc	STATUS,Z
	goto	dn
	goto	switch			;Otherwise go back to switch routine

;****************
;* Original sw=11
;****************
st3
	movf	b0_opto_now,W	;Save the current read to previous for next time
	movwf	b0_opto_prev
	xorlw	0x02			;Is it an up operation?
	btfsc	STATUS,Z
	goto	up
	movf	b0_opto_now,W
	xorlw	0x01			;Is it a down operation?
	btfsc	STATUS,Z
	goto	dn
	goto	switch			;Otherwise go back to switch routine

;*************************
;* CLICK VFO UP ONE NOTCH.
;*************************
up
	btfsc	b0_cstep,0		;Check which digit we are stepping
	incf	b0_digit0,F		; and then increment the digit
	btfsc	b0_cstep,1
	incf	b0_digit1,F
	btfsc	b0_cstep,2
	incf	b0_digit2,F
	btfsc	b0_cstep,3
	incf	b0_digit3,F
	btfsc	b0_cstep,4
	incf	b0_digit4,F
	btfsc	b0_cstep,5
	incf	b0_digit5,F
	btfsc	b0_cstep,6
	incf	b0_digit6,F
	btfsc	b0_cstep,7
	incf	b0_digit7,F

	movlw	0x0a			;Check if we overflowed the digit's decimal count
	xorwf	b0_digit0,W
	btfsc	STATUS,Z
	incf	b0_digit1,F		;Increment the next up digit if the digit overflowed
	movlw	0x0a
	xorwf	b0_digit1,W
	btfsc	STATUS,Z
	incf	b0_digit2,F
	movlw	0x0a
	xorwf	b0_digit2,W
	btfsc	STATUS,Z
	incf	b0_digit3,F
	movlw	0x0a
	xorwf	b0_digit3,W
	btfsc	STATUS,Z
	incf	b0_digit4,F
	movlw	0x0a
	xorwf	b0_digit4,W
	btfsc	STATUS,Z
	incf	b0_digit5,F
	movlw	0x0a
	xorwf	b0_digit5,W
	btfsc	STATUS,Z
	incf	b0_digit6,F
	movlw	0x0a
	xorwf	b0_digit6,W
	btfsc	STATUS,Z
	incf	b0_digit7,F

	movlw	0x0a			;Now, zero any digit that has overflowed
	xorwf	b0_digit0,W
	btfsc	STATUS,Z
	clrf	b0_digit0
	movlw	0x0a
	xorwf	b0_digit1,W
	btfsc	STATUS,Z
	clrf	b0_digit1
	movlw	0x0a
	xorwf	b0_digit2,W
	btfsc	STATUS,Z
	clrf	b0_digit2
	movlw	0x0a
	xorwf	b0_digit3,W
	btfsc	STATUS,Z
	clrf	b0_digit3
	movlw	0x0a
	xorwf	b0_digit4,W
	btfsc	STATUS,Z
	clrf	b0_digit4
	movlw	0x0a
	xorwf	b0_digit5,W
	btfsc	STATUS,Z
	clrf	b0_digit5
	movlw	0x0a
	xorwf	b0_digit6,W
	btfsc	STATUS,Z
	clrf	b0_digit6
	movlw	0x04			;Set to wrap at 39,999,999Hz when in DDS mode
	btfss	PORTA,0x03		;But, check to see if we are in DDS mode or in VFO mode
	movlw	0x0a			;We are in VFO mode, so wrap at 99,999,999Hz instead
	xorwf	b0_digit7,W
	btfsc	STATUS,Z
	clrf	b0_digit7

	call	digits_to_ddsbits	;Convert decimal digits to dds binary bits
	call	addbase			;Add base vfo, if needed
	call	shift			;Shift it into the Harris DDS numerically controlled oscillator
	call	dfreq			;Update display unit
	goto	switch			;Jump back to switch reading routine


;*************************
;* CLICK VFO Dn ONE NOTCH.
;*************************
dn
	btfsc	b0_cstep,0		;Check which digit we are stepping
	decf	b0_digit0,F		; and then decrement the digit
	btfsc	b0_cstep,1
	decf	b0_digit1,F
	btfsc	b0_cstep,2
	decf	b0_digit2,F
	btfsc	b0_cstep,3
	decf	b0_digit3,F
	btfsc	b0_cstep,4
	decf	b0_digit4,F
	btfsc	b0_cstep,5
	decf	b0_digit5,F
	btfsc	b0_cstep,6
	decf	b0_digit6,F
	btfsc	b0_cstep,7
	decf	b0_digit7,F

	movlw	0xff			;Check if we underflowed the digit's decimal count
	xorwf	b0_digit0,W
	btfsc	STATUS,Z
	decf	b0_digit1,F		;Decrement the next up digit if the digit underflowed
	movlw	0xff
	xorwf	b0_digit1,W
	btfsc	STATUS,Z
	decf	b0_digit2,F
	movlw	0xff
	xorwf	b0_digit2,W
	btfsc	STATUS,Z
	decf	b0_digit3,F
	movlw	0xff
	xorwf	b0_digit3,W
	btfsc	STATUS,Z
	decf	b0_digit4,F
	movlw	0xff
	xorwf	b0_digit4,W
	btfsc	STATUS,Z
	decf	b0_digit5,F
	movlw	0xff
	xorwf	b0_digit5,W
	btfsc	STATUS,Z
	decf	b0_digit6,F
	movlw	0xff
	xorwf	b0_digit6,W
	btfsc	STATUS,Z
	decf	b0_digit7,F

	movlw	0xff			;Now, put 9's into any digit that has underflowed
	xorwf	b0_digit0,W
	btfss	STATUS,Z
	goto	$+3
	movlw	0x09
	movwf	b0_digit0

	movlw	0xff
	xorwf	b0_digit1,W
	btfss	STATUS,Z
	goto	$+3
	movlw	0x09
	movwf	b0_digit1

	movlw	0xff
	xorwf	b0_digit2,W
	btfss	STATUS,Z
	goto	$+3
	movlw	0x09
	movwf	b0_digit2

	movlw	0xff
	xorwf	b0_digit3,W
	btfss	STATUS,Z
	goto	$+3
	movlw	0x09
	movwf	b0_digit3

	movlw	0xff
	xorwf	b0_digit4,W
	btfss	STATUS,Z
	goto	$+3
	movlw	0x09
	movwf	b0_digit4

	movlw	0xff
	xorwf	b0_digit5,W
	btfss	STATUS,Z
	goto	$+3
	movlw	0x09
	movwf	b0_digit5

	movlw	0xff
	xorwf	b0_digit6,W
	btfss	STATUS,Z
	goto	$+3
	movlw	0x09
	movwf	b0_digit6

	movlw	0xff
	xorwf	b0_digit7,W
	btfss	STATUS,Z
	goto	$+5
	movlw	0x03			;Set up to wrap back to 39,999,999Hz
	btfss	PORTA,0x03		;BUT, Check to see if we are in DDS mode or in VFO mode
	movlw	0x09			;We are in VFO mode, so wrap to 99,999,999Hz instead
	movwf	b0_digit7

	call	digits_to_ddsbits	;Convert decimal digits to dds binary bits
	call	addbase			;Add base VFO if needed
	call	shift			;Shift it into the Harris DDS numerically controlled oscillator
	call	dfreq			;Update display unit
	goto	switch			;Jump back to switch reading routine




;********************************************************************
;*                     S U B R O U T I N E S                        *
;********************************************************************

;****************************
;* WRITE LCD INSTRUCTION:
;* Use this routine to write
;* a command to the LCD.
;****************************
inst
	movwf	b0_tmp3_lo	;Save the command for now
	call	wrtlcdltch	;Write command to latch for LCD
	movf	b0_tmp3_lo,W	;Get the command back
	iorlw	0x20		;Turn LCD's CLK bit (E) on
	call	wrtlcdltch	;Now write that to the latch
	movf	b0_tmp3_lo,W	;Get the command back
							; which turns LCD's CLK bit (E) off
	call	wrtlcdltch	;Command is now in LCD
	return


;****************************
;* WRITE LCD DATA:
;* Use this routine to write
;* data to the LCD.
;****************************
write
	movwf	b0_tmp3_lo	;Save the data for now
	iorlw	0x40		;Turn on RS for LCD DATA type instructions
	call	wrtlcdltch	;Write data to latch for LCD
	movf	b0_tmp3_lo,W	;Get the data back
	iorlw	0x60		;Turn LCD's CLK bit (E) on and keep RS on
	call	wrtlcdltch	;Now write that to the latch
	movf	b0_tmp3_lo,W	;Get the data back
	iorlw	0x40		;Keep RS on and turn LCD's CLK bit (E) off
	call	wrtlcdltch	;Data is now in LCD
	return


;********************************
;* WRITE DATA TO LCD DATA LATCH:
;* Take data in accumulator and
;* write it to the LCD data port
;* latch.
;********************************
wrtlcdltch
	movwf	PORTB			;Write data to the data latch
	bsf		PORTA,0x07		;Set latch clk high
	bcf		PORTA,0x07		;Set latch clk to low, data is now latched
	return
	

;***********************
;* CLEAR LCD AND GOTO 1ST LINE:
;***********************
clearlcd
#IF LCD_TYPE == VARITRONICS
	movlw	0x08	;Home to start of first line
	call	inst	;Write to command register
	movlw	0x00	;Send the second nibble
	call	inst	;And write it
#ENDIF
#IF	LCD_TYPE == LUMEX
	movlw	0x00	;Return Home
	call	inst
	movlw	0x01
	call	inst
	call	delay_2mSec	;Delay for LCD to complete operation
#ENDIF
	return


;***********************
;* HOME LCD TO 1ST LINE:
;***********************
home1
#IF	LCD_TYPE == VARITRONICS
	movlw	0x08	;Home to start of first line
	call	inst	;Write to command register
	movlw	0x00	;Load second nibble
	call	inst	;And write it
#ENDIF
#IF	LCD_TYPE == LUMEX
	movlw	0x08	;Set DDRAM Address Command with DDRAM Address 0x00 is the start of line 1
	call	inst
	movlw	0x00
	call	inst
	call	delay_50uSec	;Delay for LCD to complete operation
#ENDIF
	return


;***********************
;* HOME LCD TO 2ND LINE:
;***********************
home2
#IF	LCD_TYPE == VARITRONICS
	movlw	0x0c	;Home to start of second line
	call	inst	;Write to command register
	movlw	0x00	;Load second nibble
	call	inst	;And write it
#ENDIF
#IF	LCD_TYPE == LUMEX
	movlw	0x0c	;Set DDRAM Address Command with DDRAM Address 0x40 is the start of line 2
	call	inst
	movlw	0x00
	call	inst
	call	delay_50uSec	;Delay for LCD to complete operation
#ENDIF
	return



;********************************
;* CONVERT DIGITS TO DDS BITS:
;* Take data in the displayed digits
;* and convert it to dds frequency control
;* bits and put them into b0_dds(3:0).
;********************************
digits_to_ddsbits		;Convert digits to dds binary bits
	clrf	b0_dds0		;First, clear out the accumulator
	clrf	b0_dds1
	clrf	b0_dds2
	clrf	b0_dds3
	;*************************************************
	pageselw	FREQ_TABLE_LOW
	;*************************************************
	movf	b0_digit0,W		;Get the one's digit
	movwf	b0_tmp1_lo		; save it into a working register
	bcf		STATUS,C		; multiply it by 4 for lookup in the freq bit table
	rlf		b0_tmp1_lo,F
	rlf		b0_tmp1_lo,F
	movf	b0_tmp1_lo,W
	call	FREQ_TABLE_O	;Get the LSB of the freq table value
	addwf	b0_dds0,F
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	movwf	b0_tmp1_lo
	call	FREQ_TABLE_O	;Get the LSB+1 of the freq table value
	addwf	b0_dds1,F
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	movwf	b0_tmp1_lo
	call	FREQ_TABLE_O	;Get the LSB+2 of the freq table value
	addwf	b0_dds2,F
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	movwf	b0_tmp1_lo
	call	FREQ_TABLE_O	;Get the MSB of the freq table value
	addwf	b0_dds3,F
	;*************************************************
	movf	b0_digit1,W		;Get the ten's digit
	movwf	b0_tmp1_lo		; save it into a working register
	bcf		STATUS,C		; multiply it by 4 for lookup in the freq bit table
	rlf		b0_tmp1_lo,F
	rlf		b0_tmp1_lo,F
	movf	b0_tmp1_lo,W
	call	FREQ_TABLE_T	;Get the LSB of the freq table value
	addwf	b0_dds0,F
	btfsc	STATUS,C		;Check for, and take care of, any carries from the addition
	incfsz	b0_dds1,F
	goto	dtd_1_1
	incfsz	b0_dds2,F
	goto	dtd_1_1
	incf	b0_dds3,F
dtd_1_1
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	movwf	b0_tmp1_lo
	call	FREQ_TABLE_T	;Get the LSB+1 of the freq table value
	addwf	b0_dds1,F
	btfsc	STATUS,C		;Check for, and take care of, any carries from the addition
	incfsz	b0_dds2,F
	goto	dtd_1_2
	incf	b0_dds3,F
dtd_1_2
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	movwf	b0_tmp1_lo
	call	FREQ_TABLE_T	;Get the LSB+2 of the freq table value
	addwf	b0_dds2,F
	btfsc	STATUS,C
	incf	b0_dds3,F
dtd_1_3
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	call	FREQ_TABLE_T	;Get the MSB of the freq table value
	addwf	b0_dds3,F
	;*************************************************
	movf	b0_digit2,W		;Get the hundred's digit
	movwf	b0_tmp1_lo		; save it into a working register
	bcf		STATUS,C		; multiply it by 4 for lookup in the freq bit table
	rlf		b0_tmp1_lo,F
	rlf		b0_tmp1_lo,F
	movf	b0_tmp1_lo,W
	call	FREQ_TABLE_H	;Get the LSB of the freq table value
	addwf	b0_dds0,F
	btfsc	STATUS,C		;Check for, and take care of, any carries from the addition
	incfsz	b0_dds1,F
	goto	dtd_2_1
	incfsz	b0_dds2,F
	goto	dtd_2_1
	incf	b0_dds3,F
dtd_2_1
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	movwf	b0_tmp1_lo
	call	FREQ_TABLE_H	;Get the LSB+1 of the freq table value
	addwf	b0_dds1,F
	btfsc	STATUS,C		;Check for, and take care of, any carries from the addition
	incfsz	b0_dds2,F
	goto	dtd_2_2
	incf	b0_dds3,F
dtd_2_2
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	movwf	b0_tmp1_lo
	call	FREQ_TABLE_H	;Get the LSB+2 of the freq table value
	addwf	b0_dds2,F
	btfsc	STATUS,C
	incf	b0_dds3,F
dtd_2_3
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	call	FREQ_TABLE_H	;Get the MSB of the freq table value
	addwf	b0_dds3,F
	;*************************************************
	movf	b0_digit3,W		;Get the thousands's digit
	movwf	b0_tmp1_lo		; save it into a working register
	bcf		STATUS,C		; multiply it by 4 for lookup in the freq bit table
	rlf		b0_tmp1_lo,F
	rlf		b0_tmp1_lo,F
	movf	b0_tmp1_lo,W
	call	FREQ_TABLE_K	;Get the LSB of the freq table value
	addwf	b0_dds0,F
	btfsc	STATUS,C		;Check for, and take care of, any carries from the addition
	incfsz	b0_dds1,F
	goto	dtd_3_1
	incfsz	b0_dds2,F
	goto	dtd_3_1
	incf	b0_dds3,F
dtd_3_1
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	movwf	b0_tmp1_lo
	call	FREQ_TABLE_K	;Get the LSB+1 of the freq table value
	addwf	b0_dds1,F
	btfsc	STATUS,C		;Check for, and take care of, any carries from the addition
	incfsz	b0_dds2,F
	goto	dtd_3_2
	incf	b0_dds3,F
dtd_3_2
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	movwf	b0_tmp1_lo
	call	FREQ_TABLE_K	;Get the LSB+2 of the freq table value
	addwf	b0_dds2,F
	btfsc	STATUS,C
	incf	b0_dds3,F
dtd_3_3
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	call	FREQ_TABLE_K	;Get the MSB of the freq table value
	addwf	b0_dds3,F
	;*************************************************
	movf	b0_digit4,W		;Get the ten's of thousands digit
	movwf	b0_tmp1_lo		; save it into a working register
	bcf		STATUS,C		; multiply it by 4 for lookup in the freq bit table
	rlf		b0_tmp1_lo,F
	rlf		b0_tmp1_lo,F
	movf	b0_tmp1_lo,W
	call	FREQ_TABLE_TK	;Get the LSB of the freq table value
	addwf	b0_dds0,F
	btfsc	STATUS,C		;Check for, and take care of, any carries from the addition
	incfsz	b0_dds1,F
	goto	dtd_4_1
	incfsz	b0_dds2,F
	goto	dtd_4_1
	incf	b0_dds3,F
dtd_4_1
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	movwf	b0_tmp1_lo
	call	FREQ_TABLE_TK	;Get the LSB+1 of the freq table value
	addwf	b0_dds1,F
	btfsc	STATUS,C		;Check for, and take care of, any carries from the addition
	incfsz	b0_dds2,F
	goto	dtd_4_2
	incf	b0_dds3,F
dtd_4_2
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	movwf	b0_tmp1_lo
	call	FREQ_TABLE_TK	;Get the LSB+2 of the freq table value
	addwf	b0_dds2,F
	btfsc	STATUS,C
	incf	b0_dds3,F
dtd_4_3
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	call	FREQ_TABLE_TK	;Get the MSB of the freq table value
	addwf	b0_dds3,F
	;*************************************************
	movf	b0_digit5,W		;Get the hundreds's of thousands digit
	movwf	b0_tmp1_lo		; save it into a working register
	btfsc	PORTA,0x03		;Check to see if we are in VFO mode
	goto	dtd_5_0			;Not in VFO mode so jump
	movlw	0x05			;We are in VFO mode, so see if the digit is >= 5
	subwf	b0_tmp1_lo,W
	btfsc	STATUS,C
	movwf	b0_tmp1_lo		;Digit was >= 5 so use the subtracted by 5 value
dtd_5_0
	bcf		STATUS,C		; multiply it by 4 for lookup in the freq bit table
	rlf		b0_tmp1_lo,F
	rlf		b0_tmp1_lo,F
	movf	b0_tmp1_lo,W
	call	FREQ_TABLE_HK	;Get the LSB of the freq table value
	addwf	b0_dds0,F
	btfsc	STATUS,C		;Check for, and take care of, any carries from the addition
	incfsz	b0_dds1,F
	goto	dtd_5_1
	incfsz	b0_dds2,F
	goto	dtd_5_1
	incf	b0_dds3,F
dtd_5_1
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	movwf	b0_tmp1_lo
	call	FREQ_TABLE_HK	;Get the LSB+1 of the freq table value
	addwf	b0_dds1,F
	btfsc	STATUS,C		;Check for, and take care of, any carries from the addition
	incfsz	b0_dds2,F
	goto	dtd_5_2
	incf	b0_dds3,F
dtd_5_2
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	movwf	b0_tmp1_lo
	call	FREQ_TABLE_HK	;Get the LSB+2 of the freq table value
	addwf	b0_dds2,F
	btfsc	STATUS,C
	incf	b0_dds3,F
dtd_5_3
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	call	FREQ_TABLE_HK	;Get the MSB of the freq table value
	addwf	b0_dds3,F
	;*************************************************

	;*************************************************
	pageselw	FREQ_TABLE_HIGH
	;*************************************************
	movf	b0_digit6,W		;Get the one's of MHz digit
	movwf	b0_tmp1_lo		; save it into a working register
	btfsc	PORTA,0x03		;Check to see if we are in VFO mode
	goto	dtd_6_0			;Not in VFO mode so jump
	clrf	b0_tmp1_lo		;We are in VFO mode, so zero this digit
dtd_6_0
	bcf		STATUS,C		; multiply it by 4 for lookup in the freq bit table
	rlf		b0_tmp1_lo,F
	rlf		b0_tmp1_lo,F
	movf	b0_tmp1_lo,W
	call	FREQ_TABLE_M	;Get the LSB of the freq table value
	addwf	b0_dds0,F
	btfsc	STATUS,C		;Check for, and take care of, any carries from the addition
	incfsz	b0_dds1,F
	goto	dtd_6_1
	incfsz	b0_dds2,F
	goto	dtd_6_1
	incf	b0_dds3,F
dtd_6_1
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	movwf	b0_tmp1_lo
	call	FREQ_TABLE_M	;Get the LSB+1 of the freq table value
	addwf	b0_dds1,F
	btfsc	STATUS,C		;Check for, and take care of, any carries from the addition
	incfsz	b0_dds2,F
	goto	dtd_6_2
	incf	b0_dds3,F
dtd_6_2
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	movwf	b0_tmp1_lo
	call	FREQ_TABLE_M	;Get the LSB+2 of the freq table value
	addwf	b0_dds2,F
	btfsc	STATUS,C
	incf	b0_dds3,F
dtd_6_3
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	call	FREQ_TABLE_M	;Get the MSB of the freq table value
	addwf	b0_dds3,F
	;*************************************************
	movf	b0_digit7,W		;Get the ten's of MHz digit
	movwf	b0_tmp1_lo		; save it into a working register
	btfsc	PORTA,0x03		;Check to see if we are in VFO mode
	goto	dtd_7_0			;Not in VFO mode so jump
	clrf	b0_tmp1_lo		;We are in VFO mode, so zero this digit
dtd_7_0
	bcf		STATUS,C		; multiply it by 4 for lookup in the freq bit table
	rlf		b0_tmp1_lo,F
	rlf		b0_tmp1_lo,F
	movf	b0_tmp1_lo,W
	call	FREQ_TABLE_TM	;Get the LSB of the freq table value
	addwf	b0_dds0,F
	btfsc	STATUS,C		;Check for, and take care of, any carries from the addition
	incfsz	b0_dds1,F
	goto	dtd_7_1
	incfsz	b0_dds2,F
	goto	dtd_7_1
	incf	b0_dds3,F
dtd_7_1
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	movwf	b0_tmp1_lo
	call	FREQ_TABLE_TM	;Get the LSB+1 of the freq table value
	addwf	b0_dds1,F
	btfsc	STATUS,C		;Check for, and take care of, any carries from the addition
	incfsz	b0_dds2,F
	goto	dtd_7_2
	incf	b0_dds3,F
dtd_7_2
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	movwf	b0_tmp1_lo
	call	FREQ_TABLE_TM	;Get the LSB+2 of the freq table value
	addwf	b0_dds2,F
	btfsc	STATUS,C
	incf	b0_dds3,F
dtd_7_3
	incf	b0_tmp1_lo,W	;Bump up for getting the next byte of freq table value
	call	FREQ_TABLE_TM	;Get the MSB of the freq table value
	addwf	b0_dds3,F
	;*************************************************
	movf	b0_dds0,W		;Now save the looked up frequency bits into the freq save memory
	movwf	b0_freq0
	movf	b0_dds1,W
	movwf	b0_freq1
	movf	b0_dds2,W
	movwf	b0_freq2
	movf	b0_dds3,W
	movwf	b0_freq3

	return




;******************
;* Set all digits to zero, dds bits to zero, and send dds bits to NCO
;******************
zero
	clrf	b0_digit0
	clrf	b0_digit1
	clrf	b0_digit2
	clrf	b0_digit3
	clrf	b0_digit4
	clrf	b0_digit5
	clrf	b0_digit6
	clrf	b0_digit7

	call	digits_to_ddsbits
	call	shift			;Shift all data out to Harris
	return




;****************************************
;* SHIFT 32 BIT DATA INTO NCO (MSb first):
;* b0_dds(3:0) contains the 32 bit data to send
;****************************************
shift
	movlw	LCNT			;Load count to shift
	movwf	b0_tmp1_lo		;
shift_1
	rlf		b0_dds0,F
	rlf		b0_dds1,F
	rlf		b0_dds2,F
	rlf		b0_dds3,F		;Carry now has bit to send
	movlw	DAT1			;Assume the bit to send to DDS is suppose to be a 1
	btfss	STATUS,C
	movlw	DAT0			;Oops, carry not set, so send a 0 instead

	call	wrtddsltch		;Setup data bit to Harris DDS chip
	iorlw	SCLK_HI			;Turn on clk bit now
	call	wrtddsltch		;Write it (clocking in valid data bit)
	andlw	SCLK_LO			;Turn off clk bit now
	call	wrtddsltch		;Write it to Harris DDS
	decfsz	b0_tmp1_lo,F	;Decrement shift count
	goto	shift_1			;Loop until all 32 data bits are sent
							;Data sent, now signal transfer complete

;***********************************
;* TRANSFER:
;* This signals the DDS that
;* a 32 bit command is complete.
;***********************************
xferit
	movlw	XFER			;Load Transfer command
	call	wrtddsltch		;Write it to DDS port

;****************************
;* Generate Harris DDS wave and return
;****************************
genit
	movlw	DDS_ON			;Load dds_on
	call	wrtddsltch		;And write it to the DDS port
	return

;********************************
;* WRITE TO DDS DATA PORT LATCH:
;* Take data in accumulator and
;* write it to the DDS data port
;* latch.
;********************************
wrtddsltch
	movwf	PORTB		;Write it to data bus
	bsf		PORTA,0x06	;Set latch clk high (data is loaded)
	bcf		PORTA,0x06	;Set latch clk low (data is latched)
	return



;***************************
;* DISPLAY:
;* Display current frequency
;***************************
dfreq
	movf	b0_digit7,W			;Get the MSD
	iorlw	0x30				;Turn it into ASCII
	movwf	b0_mfreq+0x05		;Store in 10's of MHz digit

	movf	b0_digit6,W
	iorlw	0x30
	movwf	b0_mfreq+0x06		;Store in 1's of MHz digit

	movf	b0_digit5,W
	iorlw	0x30
	movwf	b0_mfreq+0x08		;Store in 100's of KHz digit

	movf	b0_digit4,W
	iorlw	0x30
	movwf	b0_mfreq+0x09		;Store in 10's of KHz digit

	movf	b0_digit3,W
	iorlw	0x30
	movwf	b0_mfreq+0x0a		;Store in 1's of KHz digit

	movf	b0_digit2,W
	iorlw	0x30
	movwf	b0_mfreq+0x0c		;Store in 100's of Hz digit

	movf	b0_digit1,W
	iorlw	0x30
	movwf	b0_mfreq+0x0d		;Store in 10's of Hz digit

	movf	b0_digit0,W
	iorlw	0x30
	movwf	b0_mfreq+0x0e		;Store in 1's of Hz digit

;*******************************
;* Print FREQ:xx.xxx.xxx to LCD.
;*******************************
	call	home1			;Put cursor at start of 1st line
	bankisel	b0_mfreq	;Setup indirection access
	movlw	b0_mfreq
	movwf	FSR
df_1
	movlw	b0_mfreq+0x0f	;Done printing?
	xorwf	FSR,W
	btfsc	STATUS,Z
	goto	df_2
	swapf	INDF,W			;Get the character and put the high nibble in the bottom nibble
	andlw	0x0f			;Mask off
	call	write			;Write high nibble to LCD panel
	movf	INDF,W			;Get the character again
	andlw	0x0f			;Mask off
	call	write			;Write low nibble to LCD panel
	incf	FSR,F
	goto	df_1			;Loop until done with all characters
df_2
	return

	


;**************************
;* ADD BASE VFO FREQUENCY BITS TO FREQUENCY BITS.
;* Add the base vfo frequency
;* to the DDS bits.
;**************************
addbase
	btfsc	PORTA,0x03		;First check to see if we are in VFO mode
	return					;If not in VFO mode, just return, no addition needed

	movlw	0x00			;Get the vfo base frequency bits from EEPROM space
	movwf	arg1_lo
	call	read_eeprom_location	;Value returned in arg1_hi
	banksel	BANK0
	movf	arg1_hi,W
	movwf	b0_vfobase0			;Put the value into the LSB
	movlw	0x01
	movwf	arg1_lo
	call	read_eeprom_location
	banksel	BANK0
	movf	arg1_hi,W
	movwf	b0_vfobase1
	movlw	0x02
	movwf	arg1_lo
	call	read_eeprom_location
	banksel	BANK0
	movf	arg1_hi,W
	movwf	b0_vfobase2
	movlw	0x03			;Location of MSB in EEPROM space
	movwf	arg1_lo
	call	read_eeprom_location
	banksel	BANK0
	movf	arg1_hi,W
	movwf	b0_vfobase3
							;Now add the retrived vfo base value to the DDS Bits
	movf	b0_vfobase0,W
	addwf	b0_dds0,F
	btfss	STATUS,C
	goto	ab_1
	movlw	0x01
	addwf	b0_dds1,F
	btfss	STATUS,C
	goto	ab_1
	addwf	b0_dds2,F
	btfss	STATUS,C
	goto	ab_1
	addwf	b0_dds3,F
ab_1
	movf	b0_vfobase1,W
	addwf	b0_dds1,F
	btfss	STATUS,C
	goto	ab_2
	movlw	0x01
	addwf	b0_dds2,F
	btfss	STATUS,C
	goto	ab_2
	addwf	b0_dds3,F
ab_2
	movf	b0_vfobase2,W
	addwf	b0_dds2,F
	btfss	STATUS,C
	goto	ab_3
	movlw	0x01
	addwf	b0_dds3,F
ab_3
	movf	b0_vfobase3,W
	addwf	b0_dds3,F

	return



;*****************************
;* Store current stepping location to the display step memory in ascii.
;*****************************
dstep
	movlw	'0'				;First, put ascii zeros into all step display locations
	movwf	b0_mstep+0x05
	movwf	b0_mstep+0x06
	movwf	b0_mstep+0x08
	movwf	b0_mstep+0x09
	movwf	b0_mstep+0x0a
	movwf	b0_mstep+0x0c
	movwf	b0_mstep+0x0d
	movwf	b0_mstep+0x0e

	movlw	'1'
	btfss	b0_cstep,0
	goto	$+2
	movwf	b0_mstep+0x0e

	btfss	b0_cstep,1
	goto	$+2
	movwf	b0_mstep+0x0d

	btfss	b0_cstep,2
	goto	$+2
	movwf	b0_mstep+0x0c

	btfss	b0_cstep,3
	goto	$+2
	movwf	b0_mstep+0x0a

	btfss	b0_cstep,4
	goto	$+2
	movwf	b0_mstep+0x09

	btfss	b0_cstep,5
	goto	$+2
	movwf	b0_mstep+0x08

	btfss	b0_cstep,6
	goto	$+2
	movwf	b0_mstep+0x06

	btfss	b0_cstep,7
	goto	$+2
	movwf	b0_mstep+0x05

;****************************
;* Print STEP:xx.xxx.xxx to LCD.
;****************************
	call	home2			;Put cursor at start of line #2
	bankisel	b0_mstep	;Setup indirection access
	movlw	b0_mstep
	movwf	FSR

ds_1
	movlw	b0_mstep+0x0f	;Done printing?
	xorwf	FSR,W
	btfsc	STATUS,Z
	goto	ds_2
	swapf	INDF,W			;Get the character and put the high nibble in the bottom nibble
	andlw	0x0f			;Mask off
	call	write			;Write nibble to LCD panel
	movf	INDF,W			;Get the character again
	andlw	0x0f			;Mask off
	call	write
	incf	FSR,F
	goto	ds_1			;Loop until done with all characters
ds_2
	return




;**************************
;* BUMP TUNING STEPS.
;* User has pressed the VFO
;* Knob in and this routine
;* responds by changing the
;* current digit to adjust.
;**************************
bump
#if	BUMPDIR	== DOWN
	movlw	0x05			;Setup for doing a 250mSec delay
	movwf	arg2_lo
	bcf		STATUS,C
	btfsc	b0_cstep,0
	rrf		b0_cstep,F
	rrf		b0_cstep,F
	call	dstep			;Display the new step value
	call	delay_N50mSec	;Delay for switch to stop bouncing
	return
#endif
#if	BUMPDIR == UP
	movlw	0x05			;Setup for doing a 250mSec delay
	movwf	arg2_lo
	bcf		STATUS,C
	btfsc	b0_cstep,7
	rlf		b0_cstep,F
	rlf		b0_cstep,F
	call	dstep			;Display the new step value
	call	delay_N50mSec	;Delay for switch to stop bouncing
	return
#endif



;**************************
;* STORE BASE VFO
;* User has pressed the STORE BASE
;* push-button switch and this routine
;* responds by storing the current
;* dds frequency control bits as the
;* new base frequency in EEPROM
;**************************
storebase
	btfss	PORTA,0x03		;First check to make sure we are in Signal Generator mode!
	return					; If not, just return!

	movlw	0x05			;Setup for doing a 250mSec delay
	movwf	arg2_lo
	call	delay_N50mSec	;Delay a bit to debounce switch, then check again
	btfsc	PORTA,0x04		; to see that we are still requesting a "store base"
	return

	movlw	0x00			;Setup to write the LSBs of the current frequency into the base
	movwf	arg1_lo
	movf	b0_freq0,W
	movwf	arg1_hi
	call	write_eeprom_location
	banksel	BANK0
	movlw	0x01
	movwf	arg1_lo
	movf	b0_freq1,W
	movwf	arg1_hi
	call	write_eeprom_location
	banksel	BANK0
	movlw	0x02
	movwf	arg1_lo
	movf	b0_freq2,W
	movwf	arg1_hi
	call	write_eeprom_location
	banksel	BANK0
	movlw	0x03			;Setup to write the MSBs of the current frequency into the base
	movwf	arg1_lo
	movf	b0_freq3,W
	movwf	arg1_hi
	call	write_eeprom_location
	banksel	BANK0

	;Display the "NEW BASE STORED" message on line 2 (the "STEP:" line) of the LCD
	call	home2			;Put cursor at start of line #2
	clrf	b0_tmp_idx		;clear a pointer
	pageselw	base_message
sb_1
	movf	b0_tmp_idx,W
	call	base_message	;On return W has the character pointed to
	movwf	b0_tmp1_hi		;save the character for now

	swapf	b0_tmp1_hi,W	;Get the character and put the high nibble in the bottom nibble
	andlw	0x0f			;Mask off
	call	write			;Write nibble to LCD panel
	movf	b0_tmp1_hi,W	;Get the character again
	andlw	0x0f			;Mask off
	call	write
	incf	b0_tmp_idx,F
	movlw	0x0f			;Check to see if we are done
	xorwf	b0_tmp_idx,W
	btfss	STATUS,Z
	goto	sb_1			;Loop until done with all 15 characters of the message

	movlw	0x01			;Setup for doing 50mSec delays
	movwf	arg2_lo
sb_2
	btfss	PORTA,0x04		;Now wait around for the switch to be released and debounced
	goto	sb_2
	call	delay_N50mSec
	btfss	PORTA,0x04
	goto	sb_2
	call	delay_N50mSec
	btfss	PORTA,0x04
	goto	sb_2

	movlw	0x05			;Setup for doing a 250mSec delay
	movwf	arg2_lo
	call	delay_N50mSec

	;Restore the "STEP:" message line
	call	dstep
	return


	end
;End of File (vfo_main.asm)
