// ******************************************************
// ******************************************************
//
//  Main Project     : OCRG
//  Project Name     : OCRG WDPC - WatchDog Power Cycler
//  File Name        : main.c 
//  Version          : x0.1
//  PIC Used         : PIC16F648A
//  Compiler         : C2C plus C-Compiler
//  Assembler        : MPLAB MPASMWIN from Microchip Technology, Inc.
//  Initial Date     : Aug 28, 2008
//  Last Update date : Aug 28, 2008 0.1
//  Tab size         : 3
//  Firmware Eng.    : WA7ZVY
//  HW Design Eng.   : WA7ZVY
// 
// ******************************************************
//
//  Description of this C code:
//
//  This code starts execution upon the release of the
//  power on reset condition. The code continually
//  executes the main routine and executes the
//  interrupt code upon an interrupting condition.
//
//		1) interrupt dispatcher
//		2) main loop
//
// ******************************************************
//
//  Function List (in top-2-bottom order):
//    void interrupt(void);
//    void main(void);
//
// ******************************************************
//
//    Modification History:
//    =-=-=-=-=-=-=-=-=-=-=
//    08-28-08 pm - Initial creation.
//
// ******************************************************

// *****************************************************
//    === Include files ===
// *****************************************************
#include	"p16f648a.h";
#include	"constants.h";
#include	"micro_bits.h";
//#include "main.h";
#include	"init_hardware.h"
#include	"comm.h"



// *****************************************************
//    === Compilier Pragma Statements ===
// *****************************************************
#pragma CLOCK_FREQ		4000000

// *****************************************************
//    == Compilier Conditional Compile Switches ===
// *****************************************************
//
//    There are two compiler switches - DEBUG and SIM.
//
//    DEBUG does nothing in this code as it is always in
//		debug mode.
//
//    SIM configures the compile for execution on the
//     ScourceBoost IDE debug environment.
//
// *****************************************************
//#define	DEBUG
//#define	SIM

// *****************************************************
//        === Configuration Word Definitions ===
// *****************************************************
// Defines for 16F648A operation in the WDPC Board
asm __config _INTRC_OSC_NOCLK & _CP_OFF & _DATA_CP_OFF & _LVP_OFF & _BOREN_OFF & _PWRTE_ON & _WDT_OFF & _MCLRE_OFF

// *****************************************************
//    === Global Variables ===
// *****************************************************
//	Define FLASH ROM resident copyright statement
const	char	*copyright =
	"Copyright(c)2008, WA7ZVY";

//	Define the Morse Code Sign-On Text
//	Format is: "-" = Dash, "." = Dot, " " = Character Space
const	char	*signon =
//  "OCRG WDPC V1.0";
  "     ---  -.-.  .-.  --.     .--  -..  .--.  -.-.     ...-  .----  .-.-.-  -----     ";
// 000000000111111111122222222223333333333444444444455555555556666666666777777777788888888889999999999
// 123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789
#define	SIGNON_LENGTH	0x55

//  " V  ";
//  " ...-  ";
// 000000000111111111122222222223333333333444444444455555555556666666666777777777788888888889999999999
// 123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789
//#define	SIGNON_LENGTH	0x07


//	Define various global variables
char	count50msec;
char	count1sec;
char	countpulse;
char	HeartBeat;
char	PreviousHeartBeat;
char	HeartBeatToggled;
char	ReBoot;
char	aliveblink;
int		blinktime;


// *****************************************************
// *****************************************************
//
// 	Name:		interrupt
// 	
// 	Abstract:
// 		This is the interrupt handler. It is entered
//		whenever there is an interrupt asserted.
//		It looks for the source of the interrupt
//		and dispatches to the appropriate interrupt
//		handling routine.
//		This routine is the top level interrupt code (ie. it
//		calls other routines which return to it, but
//		it is never called by other routines).
//
//		Sources of interrupts are:
//			1). Timer 0
//				a). Not used
//
//			2). Timer 1
//				a). Used for 50 millisecond tick generation.
//
//			3). Timer 2
//				a). Not used
// 		
// 	Global Variables:
// 		
// 	Local Variables:
//
// 	Return Values:	None.
// 		
// *****************************************************
// *****************************************************


void interrupt(void)
{
// Check for any pending interrupts and process them here
//

// Timer 2 Source is not used.

// Timer 1 (50 millisecond tick processing)
	if(test_bit_HI(pir1,TMR1IF))
	{
		countpulse++;						// increment the heartbeat pulse counter
		count50msec++;						// increment the number of 50 millisecond time intervals that have elapsed
		if(count50msec >= 0x14)
		{
			count50msec = 0x00;				// zero the number of 50 millisecond time intervals that have elapsed
			count1sec++;					// increment the number of 1 second time intervals that have elapsed
		}
		tmr1l = TICKTIMELO1;
		tmr1h = TICKTIMEHI1;
		clear_bit(pir1,TMR1IF);
	}

// Timer 0 Source is not used.

}



// *****************************************************
// *****************************************************
//
// 	Name:		main
// 	
// 	Abstract:
// 		This is the main entry point for
//		code execution on the WDPC board.
//		This routine is the highest level code (ie. it
//		calls other routines which return to it, but
//		it is never called by other routines).
// 		
// 	Global Variables:
// 		
// 	Local Variables:
// 	
// 	Return Values:	None.
// 		
// *****************************************************
// *****************************************************
void main(void)
{
	init_hardware();
	
	//Initialize various global variables
	countpulse = 0x00;
	count50msec = 0x00;
	count1sec = 0x00;
	PreviousHeartBeat = 0x00;
	HeartBeatToggled = FALSE;
	ReBoot = TRUE;				// Force a ReBoot to occur to get things initially started

	//Initialize the timer registers
	tmr1l = TICKTIMELO1;
	tmr1h = TICKTIMEHI1;
	
	// When we get to here, the signon morse code message is sent via the LEDs
	char	str_pos;
	for(str_pos = 0; str_pos < SIGNON_LENGTH; str_pos++)
	{
		send_morse_element(signon[str_pos]);	//send the morse code character	element (dash, dot, or character space)
	}

	// Turn on timer 1
	set_bit(pie1,TMR1IE);						// Enable Timer 1 Interrupt
	enable_interrupt(PEIE);						// Enable the Peripheral Interrupts
	asm nop;									// delay because of following accesses to same register
	enable_interrupt(GIE);						// GLOBAL ENABLE OF ALL OF
	asm nop;									//  THE ABOVE INTERRUPTS!

	set_bit(t1con,TMR1ON);
	
	
//idle_loop:
	while(TRUE)
	{
		
		// Determine if the incoming Heartbeat is toggling (it should be a 1Hz signal)
		disable_interrupt(GIE);	// Mask all interrupts while doing the following tests
		disable_interrupt(GIE);
		HeartBeat = portb & HEARTBEAT_MASK;
		if(countpulse < DEADCOUNT)
		{
			if(HeartBeat != PreviousHeartBeat)
				HeartBeatToggled = TRUE;
			if(test_bit_LO(portb,BYPASS_L_SIG))		// Or, if the bypass switch is in bypass mode
				HeartBeatToggled = TRUE;			//  then always force the HeartBeatToggled true
		}
		if((countpulse >= DOOMCOUNT) && (HeartBeatToggled == FALSE))
			clear_bit(porta,PENDING_DOOM_LED);			// turn on the Pending Doom LED
		if((countpulse >= DOOMCOUNT) && (HeartBeatToggled == TRUE))
			set_bit(porta,PENDING_DOOM_LED);			// turn off the Pending Doom LED
			
		if((countpulse >= DEADCOUNT) && (HeartBeatToggled == FALSE))
			ReBoot = TRUE;
		if((countpulse >= DEADCOUNT) && (HeartBeatToggled == TRUE))
		{
			countpulse = 0x00;
			PreviousHeartBeat = HeartBeat;
			HeartBeatToggled = FALSE;
		}
		enable_interrupt(GIE);	// Turn interrupts back on
		asm nop;
		asm nop;
		
		if(ReBoot == TRUE)
		{
			clear_bit(porta,PENDING_DOOM_LED);			// turn on Pending Doom LED
			asm nop;
			TimeDelay(PENDING_DOOM_DELAY);
			set_bit(porta,AC_POWER_OFF_LED);	 		// turn off all other LEDs
			asm nop;
			set_bit(porta,AC_POWER_ON_LED);
			asm nop;
			set_bit(porta,BOOT_WAIT_LED);
			asm nop;
			set_bit(porta,WATCHDOG_ARMED_LED);
			asm nop;
			set_bit(porta,WATCHDOG_ALIVE_LED);
			asm nop;
			TimeDelay(PENDING_DOOM_DELAY);
			
			set_bit(portb,AC_POWER_OFF_SIG);			// turn off AC power
			asm nop;
			set_bit(porta,PENDING_DOOM_LED);			// turn off Pending Doom LED
			asm nop;
			clear_bit(porta,AC_POWER_OFF_LED);			// turn on AC Power Off LED
			asm nop;
			TimeDelay(POWER_OFF_DELAY);
			
			set_bit(porta,AC_POWER_OFF_LED);			// turn off AC Power Off LED;
			asm nop;
			clear_bit(portb,AC_POWER_OFF_SIG);			// turn on AC power
			asm nop;
			clear_bit(porta,AC_POWER_ON_LED);			// turn on AC Power On LED;
			asm nop;
			
			if(test_bit_HI(portb,BYPASS_L_SIG))			// if the bypass switch is in normal mode (not bypassed)
			{
				TimeDelay(AC_POWER_ON_DELAY);
				clear_bit(porta,BOOT_WAIT_LED);			// turn on Boot Wait LED
				asm nop;
				TimeDelay(BOOT_WAIT_DELAY);
			
				set_bit(porta,BOOT_WAIT_LED);			// turn off Boot Wait LED
				asm nop;
				clear_bit(porta,WATCHDOG_ARMED_LED);	// turn on WatchDog Armed LED
				asm nop;
			}
			
			ReBoot = FALSE;
			countpulse = 0x00;
			HeartBeat = portb & HEARTBEAT_MASK;
			PreviousHeartBeat = HeartBeat;
		}

		else
		{
			++blinktime;									// increment the blink time counter
			if(blinktime >= BLINKRATE)
			{
				blinktime = 0x0000;
				aliveblink = ((aliveblink ^ 0x01) & 0x01);	// invert the WatchDog Alive LED
				if(aliveblink == 0x00)
					clear_bit(porta,WATCHDOG_ALIVE_LED);
				if(aliveblink == 0x01)
					set_bit(porta,WATCHDOG_ALIVE_LED);
			if(test_bit_LO(portb,BYPASS_L_SIG))				// if the bypass switch is in bypass mode
				set_bit(porta,WATCHDOG_ARMED_LED);			//  then turn off the WatchDog Armed LED
			asm nop;
			if(test_bit_HI(portb,BYPASS_L_SIG))				// if the bypass switch is in normal mode (not bypassed)
				clear_bit(porta,WATCHDOG_ARMED_LED);		//  then turn on the WatchDog Armed LED
			}
		}
	}
}


// End of File (main.c)
